/******************************************************************
Copyright 2000 by Object Craft P/L, Melbourne, Australia.

                        All Rights Reserved

Permission to use, copy, modify, and distribute this software and its
documentation for any purpose and without fee is hereby granted,
provided that the above copyright notice appear in all copies and that
both that copyright notice and this permission notice appear in
supporting documentation, and that the name of Object Craft
is not be used in advertising or publicity pertaining to
distribution of the software without specific, written prior
permission.

OBJECT CRAFT DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
EVENT SHALL OBJECT CRAFT BE LIABLE FOR ANY SPECIAL, INDIRECT OR
CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF
USE, DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR
OTHER TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
PERFORMANCE OF THIS SOFTWARE.

******************************************************************/

#include "paint.h"

void make_rect_vpath(ArtVpath *vpath, double x1, double y1, double x2, double y2)
{
    vpath->code = ART_MOVETO;
    vpath->x = x1;
    vpath->y = y1;
    vpath++;
    vpath->code = ART_LINETO;
    vpath->x = x2;
    vpath->y = y1;
    vpath++;
    vpath->code = ART_LINETO;
    vpath->x = x2;
    vpath->y = y2;
    vpath++;
    vpath->code = ART_LINETO;
    vpath->x = x1;
    vpath->y = y2;
    vpath++;
    vpath->code = ART_LINETO;
    vpath->x = x1;
    vpath->y = y1;
    vpath++;
    vpath->code = ART_END;
    vpath->x = 0;
    vpath->y = 0;
}

static PathObj *new_PathObj(ArtVpath *vpath, int vpath_len);

static char path_dash__doc__[] = 
"dash(offset, dashlen_list)\n"
"\n"
"Dash the path starting at offset using the dash lengths in dashlen_list.";

static PyObject *path_dash(PathObj *self, PyObject *args)
{
    ArtVpathDash dash;
    PyObject *seq;
    int i;
    ArtVpath *new;

    static char *dash_err = "dashlen_list must be a sequence of numbers";

    if (!PyArg_ParseTuple(args, "dO", &dash.offset, &seq))
	return NULL;
    if (!PySequence_Check(seq)) {
	set_error(PyExc_TypeError, dash_err);
	return NULL;
    }
    dash.n_dash = PySequence_Length(seq);
    if (dash.n_dash == 0) {
	set_error(PyExc_TypeError, dash_err);
	return NULL;
    }
    dash.dash = Py_Malloc(dash.n_dash * sizeof(*dash.dash));
    for (i = 0; i < dash.n_dash; i++) {
	PyObject *len = PySequence_GetItem(seq, i);
	Py_DECREF(len);
	if (!PyNumber_Check(len)) {
	    PyMem_DEL(dash.dash);
	    set_error(PyExc_TypeError, dash_err);
	    return NULL;
	}
	dash.dash[i] = PyFloat_AsDouble(len);
    }
    new = art_vpath_dash(self->vpath, &dash);
    return (PyObject*)new_PathObj(new, self->vpath_len);
}

static char path_transform__doc__[] = 
"transform(affine)\n"
"\n"
"Apply the affine transformation to the path returning a new path";

static PyObject *path_transform(PathObj *self, PyObject *args)
{
    AffineObj *affine_obj;
    ArtVpath *new;

    if (!PyArg_ParseTuple(args, "O!",
			  &AffineType, (PyObject*)&affine_obj))
	return NULL;

    new = art_vpath_affine_transform(self->vpath, affine_obj->affine);
    return (PyObject*)new_PathObj(new, self->vpath_len);
}

static struct PyMethodDef path_methods[] = {
    { "dash", (PyCFunction)path_dash, METH_VARARGS, path_dash__doc__ },
    { "transform", (PyCFunction)path_transform, METH_VARARGS, path_transform__doc__ },

    { NULL, NULL }		/* sentinel */
};

static PathObj *new_PathObj(ArtVpath *vpath, int vpath_len)
{
    PathObj *self;

    self = PyObject_NEW(PathObj, &PathType);
    if (self == NULL)
	return NULL;
    self->vpath = vpath;
    self->vpath_len = vpath_len;
    return self;
}

static void dealloc_PathObj(PathObj *self)
{
    if (self->vpath != NULL)
	art_free(self->vpath);
    PyMem_DEL(self);
}

static PyObject *path_getattr(ImageObj *self, char *name)
{
    return Py_FindMethod(path_methods, (PyObject *)self, name);
}

static char PathType__doc__[] = 
"";

PyTypeObject PathType = {
    PyObject_HEAD_INIT(&PyType_Type)
    0,				/*ob_size*/
    "Path",			/*tp_name*/
    sizeof(PathObj),		/*tp_basicsize*/
    0,				/*tp_itemsize*/
    /* methods */
    (destructor)dealloc_PathObj, /*tp_dealloc*/
    (printfunc)0,		/*tp_print*/
    (getattrfunc)path_getattr,	/*tp_getattr*/
    (setattrfunc)0,		/*tp_setattr*/
    (cmpfunc)0,			/*tp_compare*/
    (reprfunc)0,		/*tp_repr*/
    0,				/*tp_as_number*/
    0,				/*tp_as_sequence*/
    0,				/*tp_as_mapping*/
    (hashfunc)0,		/*tp_hash*/
    (ternaryfunc)0,		/*tp_call*/
    (reprfunc)0,		/*tp_str*/

    /* Space for future expansion */
    0L, 0L, 0L, 0L,
    PathType__doc__ /* Documentation string */
};

PyObject *path_make_path(PyObject *args)
{
    PyObject *path;
    int i, path_len, vpath_len;
    ArtVpath *vpath;
    PyObject *item, *code;

    static char *path_err = "path must be a sequence of (code, x, y) tuples";

    if (!PyArg_ParseTuple(args, "O", &path))
	return NULL;
    if (!PySequence_Check(path)) {
	set_error(PyExc_TypeError, path_err);
	return NULL;
    }

    path_len = PySequence_Length(path);
    if (path_len == 0) {
	Py_INCREF(Py_None);
	return Py_None;
    }

    item = PySequence_GetItem(path, path_len - 1); /* new ref */
    Py_DECREF(item);
    if (!PyTuple_Check(item)
	|| PyTuple_Size(item) != 3
	|| !PyInt_Check((code = PyTuple_GetItem(item, 0)))) {
	set_error(PyExc_TypeError, path_err);
	return NULL;
    }
    if (PyInt_AsLong(code) != ART_END)
	vpath_len = path_len + 1;
    else
	vpath_len = path_len;
    vpath = Py_Malloc(vpath_len * sizeof(*vpath));
    if (vpath_len != path_len) {
	vpath[path_len].code = ART_END;
	vpath[path_len].x = 0;
	vpath[path_len].y = 0;
    }

    for (i = 0; i < path_len; i++) {
	PyObject *x, *y;

	item = PySequence_GetItem(path, i); /* new ref */
	Py_DECREF(item);
	if (!PyTuple_Check(item)
	    || PyTuple_Size(item) != 3
	    || !PyInt_Check((code = PyTuple_GetItem(item, 0)))
	    || !PyNumber_Check((x = PyTuple_GetItem(item, 1)))
	    || !PyNumber_Check((y = PyTuple_GetItem(item, 2)))) {
	    PyMem_DEL(vpath);
	    set_error(PyExc_TypeError, path_err);
	    return NULL;
	}
	vpath[i].code = PyInt_AsLong(code);
	vpath[i].x = PyFloat_AsDouble(x);
	vpath[i].y = PyFloat_AsDouble(y);
    }

    return (PyObject*)new_PathObj(vpath, vpath_len);
}

static PyObject *build_arc(int is_pie, PyObject *args)
{
    double x1, y1, x2, y2, width, height, center_x, center_y;
    int num_points, i, idx;
    double angle, sweep;
    double circ, theta, delta_theta;
    ArtVpath *path;

    if (!PyArg_ParseTuple(args, "dddddd", &x1, &y1, &x2, &y2, &angle, &sweep))
	return NULL;

    width = x2 - x1;
    height = y2 - y1;
    if (width > height)
        circ = M_PI * width;
    else
        circ = M_PI * height;
    circ = circ * sweep / 360.0;

    num_points = circ / 4;
    if (num_points < 5)
        num_points = 5;

    center_x = (x1 + x2) / 2;
    center_y = (y1 + y2) / 2;
    if (is_pie) {
	path = art_new(ArtVpath, num_points + 3);
	path[0].code = ART_MOVETO;
	path[0].x = center_x;
	path[0].y = center_y;
	idx = 1;
    } else {
	path = art_new(ArtVpath, num_points + 1);
	idx = 0;
    }
    theta = M_PI * angle / 180.0;
    delta_theta = (M_PI * sweep / 180.0) / (num_points - 1);
    for (i = 0; i < num_points; i++, idx++) {
	if (idx == 0)
	    path[idx].code = ART_MOVETO;
	else
	    path[idx].code = ART_LINETO;
	path[idx].x = center_x + width / 2.0 * cos(theta);
	path[idx].y = center_y - height / 2.0 * sin(theta);
        theta = theta + delta_theta;
    }
    if (is_pie) {
	path[idx].code = ART_LINETO;
	path[idx].x = center_x;
	path[idx].y = center_y;
	idx++;
    }
    path[idx].code = ART_END;
    path[idx].x = 0;
    path[idx].y = 0;

    return (PyObject*)new_PathObj(path, num_points + 3);
}

PyObject *path_make_arc(PyObject *args)
{
    return build_arc(0, args);
}

PyObject *path_make_arc_pie(PyObject *args)
{
    return build_arc(1, args);
}

PyObject *path_make_rect(PyObject *args)
{
    double x1, y1, x2, y2;
    ArtVpath *path;

    if (!PyArg_ParseTuple(args, "dddd", &x1, &y1, &x2, &y2))
	return NULL;

    path = art_new(ArtVpath, 6);
    make_rect_vpath(path, x1, y1, x2, y2);

    return (PyObject*)new_PathObj(path, 6);
}

PyObject *path_make_line(PyObject *args)
{
    double x1, y1, x2, y2;
    ArtVpath *path;

    if (!PyArg_ParseTuple(args, "dddd", &x1, &y1, &x2, &y2))
	return NULL;

    path = art_new(ArtVpath, 3);
    path[0].code = ART_MOVETO;
    path[0].x = x1;
    path[0].y = y1;
    path[1].code = ART_LINETO;
    path[1].x = x2;
    path[1].y = y2;
    path[2].code = ART_END;
    path[2].x = 0;
    path[2].y = 0;

    return (PyObject*)new_PathObj(path, 3);
}
